<?php
require_once 'config.php';

if(!isLoggedIn()) {
    header("Location: login.php");
    exit();
}

$user_id = $_SESSION['user_id'];

// Get filter parameters
$type_filter = isset($_GET['type']) ? sanitize($_GET['type']) : '';
$status_filter = isset($_GET['status']) ? sanitize($_GET['status']) : '';
$search = isset($_GET['search']) ? sanitize($_GET['search']) : '';
$start_date = isset($_GET['start_date']) ? sanitize($_GET['start_date']) : '';
$end_date = isset($_GET['end_date']) ? sanitize($_GET['end_date']) : '';
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$per_page = 15;
$offset = ($page - 1) * $per_page;

// Build query
$where_conditions = ["user_id = $user_id"];
$params = [];
$types = "";

if($type_filter && $type_filter != 'all') {
    $where_conditions[] = "type = ?";
    $params[] = $type_filter;
    $types .= "s";
}

if($status_filter && $status_filter != 'all') {
    $where_conditions[] = "status = ?";
    $params[] = $status_filter;
    $types .= "s";
}

if($search) {
    $where_conditions[] = "(description LIKE ? OR id LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $types .= "ss";
}

if($start_date) {
    $where_conditions[] = "DATE(created_at) >= ?";
    $params[] = $start_date;
    $types .= "s";
}

if($end_date) {
    $where_conditions[] = "DATE(created_at) <= ?";
    $params[] = $end_date;
    $types .= "s";
}

$where_sql = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";

// Get total transactions
$count_sql = "SELECT COUNT(*) as total FROM transactions $where_sql";
$count_stmt = $conn->prepare($count_sql);
if($types) $count_stmt->bind_param($types, ...$params);
$count_stmt->execute();
$total_result = $count_stmt->get_result();
$total_transactions = $total_result->fetch_assoc()['total'];
$total_pages = ceil($total_transactions / $per_page);

// Get transactions
$transactions_sql = "SELECT * FROM transactions 
                     $where_sql 
                     ORDER BY created_at DESC 
                     LIMIT ? OFFSET ?";
$params[] = $per_page;
$params[] = $offset;
$types .= "ii";

$transactions_stmt = $conn->prepare($transactions_sql);
$transactions_stmt->bind_param($types, ...$params);
$transactions_stmt->execute();
$transactions_result = $transactions_stmt->get_result();

// Get statistics
$stats = $conn->query("
    SELECT 
        SUM(CASE WHEN type = 'deposit' AND status = 'completed' THEN amount ELSE 0 END) as total_deposits,
        SUM(CASE WHEN type = 'withdrawal' AND status = 'completed' THEN amount ELSE 0 END) as total_withdrawals,
        SUM(CASE WHEN type = 'activation' THEN amount ELSE 0 END) as total_activation,
        SUM(CASE WHEN type = 'referral_bonus' AND status = 'completed' THEN amount ELSE 0 END) as total_referral_bonus,
        SUM(CASE WHEN type = 'product_purchase' THEN amount ELSE 0 END) as total_purchases,
        COUNT(*) as total_count
    FROM transactions 
    WHERE user_id = $user_id
")->fetch_assoc();

// Get counts by type
$type_counts = $conn->query("
    SELECT type, COUNT(*) as count 
    FROM transactions 
    WHERE user_id = $user_id 
    GROUP BY type
")->fetch_all(MYSQLI_ASSOC);

// Get counts by status
$status_counts = $conn->query("
    SELECT status, COUNT(*) as count 
    FROM transactions 
    WHERE user_id = $user_id 
    GROUP BY status
")->fetch_all(MYSQLI_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Transactions - Project Sale</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css">
    <style>
        .transaction-row {
            transition: all 0.2s ease;
        }
        .transaction-row:hover {
            background-color: #f9fafb;
        }
        .filter-tab {
            transition: all 0.3s ease;
        }
        .filter-tab:hover {
            transform: scale(1.05);
        }
        .stat-card {
            transition: all 0.3s ease;
        }
        .stat-card:hover {
            transform: translateY(-5px);
        }
    </style>
</head>
<body class="bg-gray-50">
    <!-- Include Header -->
    <?php include 'includes/header.php'; ?>
    
    <div class="flex min-h-screen">
        <!-- Sidebar -->
        <div class="hidden md:block w-64 bg-white shadow-lg">
            <?php include 'includes/sidebar.php'; ?>
        </div>
        
        <!-- Main Content -->
        <div class="flex-1 p-6">
            <!-- Breadcrumb -->
            <div class="mb-6">
                <nav class="flex" aria-label="Breadcrumb">
                    <ol class="inline-flex items-center space-x-1 md:space-x-3">
                        <li class="inline-flex items-center">
                            <a href="index.php" class="inline-flex items-center text-sm text-gray-700 hover:text-blue-600">
                                <i class="fas fa-home mr-2"></i> Home
                            </a>
                        </li>
                        <li>
                            <div class="flex items-center">
                                <i class="fas fa-chevron-right text-gray-400 text-xs"></i>
                                <a href="dashboard.php" class="ml-1 text-sm text-gray-700 hover:text-blue-600 md:ml-2">Dashboard</a>
                            </div>
                        </li>
                        <li aria-current="page">
                            <div class="flex items-center">
                                <i class="fas fa-chevron-right text-gray-400 text-xs"></i>
                                <span class="ml-1 text-sm text-gray-500 md:ml-2 font-medium">Transactions</span>
                            </div>
                        </li>
                    </ol>
                </nav>
            </div>
            
            <!-- Header -->
            <div class="mb-8">
                <h1 class="text-3xl font-bold text-gray-800 mb-2">
                    <i class="fas fa-exchange-alt mr-2"></i> Transaction History
                </h1>
                <p class="text-gray-600">Track all your financial activities</p>
            </div>
            
            <!-- Statistics Cards -->
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
                <div class="stat-card bg-gradient-to-r from-green-500 to-emerald-600 rounded-xl shadow-lg p-6 text-white animate__animated animate__fadeInUp">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-green-100 text-sm">Total Deposits</p>
                            <h3 class="text-3xl font-bold mt-2">৳<?php echo number_format($stats['total_deposits'] ?? 0, 2); ?></h3>
                        </div>
                        <div class="w-14 h-14 bg-white/20 rounded-full flex items-center justify-center">
                            <i class="fas fa-plus-circle text-2xl"></i>
                        </div>
                    </div>
                    <div class="mt-4">
                        <a href="deposit.php" class="text-green-100 hover:text-white text-sm font-medium">
                            <i class="fas fa-arrow-right mr-1"></i> Deposit More
                        </a>
                    </div>
                </div>
                
                <div class="stat-card bg-gradient-to-r from-red-500 to-pink-600 rounded-xl shadow-lg p-6 text-white animate__animated animate__fadeInUp" style="animation-delay: 0.1s">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-red-100 text-sm">Total Withdrawals</p>
                            <h3 class="text-3xl font-bold mt-2">৳<?php echo number_format($stats['total_withdrawals'] ?? 0, 2); ?></h3>
                        </div>
                        <div class="w-14 h-14 bg-white/20 rounded-full flex items-center justify-center">
                            <i class="fas fa-minus-circle text-2xl"></i>
                        </div>
                    </div>
                    <div class="mt-4">
                        <a href="withdraw.php" class="text-red-100 hover:text-white text-sm font-medium">
                            <i class="fas fa-arrow-right mr-1"></i> Withdraw Now
                        </a>
                    </div>
                </div>
                
                <div class="stat-card bg-gradient-to-r from-blue-500 to-indigo-600 rounded-xl shadow-lg p-6 text-white animate__animated animate__fadeInUp" style="animation-delay: 0.2s">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-blue-100 text-sm">Referral Earnings</p>
                            <h3 class="text-3xl font-bold mt-2">৳<?php echo number_format($stats['total_referral_bonus'] ?? 0, 2); ?></h3>
                        </div>
                        <div class="w-14 h-14 bg-white/20 rounded-full flex items-center justify-center">
                            <i class="fas fa-users text-2xl"></i>
                        </div>
                    </div>
                    <div class="mt-4">
                        <a href="referrals.php" class="text-blue-100 hover:text-white text-sm font-medium">
                            <i class="fas fa-arrow-right mr-1"></i> View Referrals
                        </a>
                    </div>
                </div>
                
                <div class="stat-card bg-gradient-to-r from-purple-500 to-pink-600 rounded-xl shadow-lg p-6 text-white animate__animated animate__fadeInUp" style="animation-delay: 0.3s">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-purple-100 text-sm">Total Transactions</p>
                            <h3 class="text-3xl font-bold mt-2"><?php echo number_format($stats['total_count'] ?? 0); ?></h3>
                        </div>
                        <div class="w-14 h-14 bg-white/20 rounded-full flex items-center justify-center">
                            <i class="fas fa-list text-2xl"></i>
                        </div>
                    </div>
                    <div class="mt-4">
                        <a href="transactions.php" class="text-purple-100 hover:text-white text-sm font-medium">
                            <i class="fas fa-arrow-right mr-1"></i> View All
                        </a>
                    </div>
                </div>
            </div>
            
            <!-- Filter Tabs -->
            <div class="mb-8">
                <div class="flex flex-wrap gap-3">
                    <!-- Type Filters -->
                    <div class="bg-white rounded-lg shadow p-4 flex-1 min-w-[250px]">
                        <h4 class="font-bold text-gray-800 mb-3">
                            <i class="fas fa-filter text-blue-500 mr-2"></i> Filter by Type
                        </h4>
                        <div class="flex flex-wrap gap-2">
                            <a href="transactions.php<?php echo $status_filter ? '?status=' . $status_filter : ''; ?><?php echo $search ? '&search=' . urlencode($search) : ''; ?>"
                               class="filter-tab px-3 py-2 rounded-lg text-sm font-medium <?php echo !$type_filter || $type_filter == 'all' ? 'bg-blue-100 text-blue-700' : 'bg-gray-100 text-gray-700 hover:bg-gray-200'; ?>">
                                All Types
                            </a>
                            <?php foreach($type_counts as $type): ?>
                            <a href="transactions.php?type=<?php echo $type['type']; ?><?php echo $status_filter ? '&status=' . $status_filter : ''; ?><?php echo $search ? '&search=' . urlencode($search) : ''; ?>"
                               class="filter-tab px-3 py-2 rounded-lg text-sm font-medium <?php echo $type_filter == $type['type'] ? 'bg-blue-100 text-blue-700' : 'bg-gray-100 text-gray-700 hover:bg-gray-200'; ?>">
                                <?php echo ucfirst(str_replace('_', ' ', $type['type'])); ?> (<?php echo $type['count']; ?>)
                            </a>
                            <?php endforeach; ?>
                        </div>
                    </div>
                    
                    <!-- Status Filters -->
                    <div class="bg-white rounded-lg shadow p-4 flex-1 min-w-[250px]">
                        <h4 class="font-bold text-gray-800 mb-3">
                            <i class="fas fa-tasks text-green-500 mr-2"></i> Filter by Status
                        </h4>
                        <div class="flex flex-wrap gap-2">
                            <a href="transactions.php<?php echo $type_filter ? '?type=' . $type_filter : ''; ?><?php echo $search ? '&search=' . urlencode($search) : ''; ?>"
                               class="filter-tab px-3 py-2 rounded-lg text-sm font-medium <?php echo !$status_filter || $status_filter == 'all' ? 'bg-green-100 text-green-700' : 'bg-gray-100 text-gray-700 hover:bg-gray-200'; ?>">
                                All Status
                            </a>
                            <?php foreach($status_counts as $status): ?>
                            <a href="transactions.php<?php echo $type_filter ? '?type=' . $type_filter : '?'; ?>&status=<?php echo $status['status']; ?><?php echo $search ? '&search=' . urlencode($search) : ''; ?>"
                               class="filter-tab px-3 py-2 rounded-lg text-sm font-medium <?php echo $status_filter == $status['status'] ? 'bg-green-100 text-green-700' : 'bg-gray-100 text-gray-700 hover:bg-gray-200'; ?>">
                                <?php echo ucfirst($status['status']); ?> (<?php echo $status['count']; ?>)
                            </a>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Search and Filter -->
            <div class="bg-white rounded-xl shadow-lg p-6 mb-8 animate__animated animate__fadeIn">
                <form method="GET" action="" class="space-y-4">
                    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                        <!-- Search -->
                        <div>
                            <label class="block text-gray-700 text-sm font-bold mb-2">
                                <i class="fas fa-search text-blue-500 mr-2"></i> Search
                            </label>
                            <input type="text" name="search" value="<?php echo htmlspecialchars($search); ?>"
                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300"
                                   placeholder="Search transactions...">
                        </div>
                        
                        <!-- Type Filter -->
                        <div>
                            <label class="block text-gray-700 text-sm font-bold mb-2">
                                Type
                            </label>
                            <select name="type" 
                                    class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300">
                                <option value="">All Types</option>
                                <option value="deposit" <?php echo $type_filter == 'deposit' ? 'selected' : ''; ?>>Deposit</option>
                                <option value="withdrawal" <?php echo $type_filter == 'withdrawal' ? 'selected' : ''; ?>>Withdrawal</option>
                                <option value="activation" <?php echo $type_filter == 'activation' ? 'selected' : ''; ?>>Activation</option>
                                <option value="referral_bonus" <?php echo $type_filter == 'referral_bonus' ? 'selected' : ''; ?>>Referral Bonus</option>
                                <option value="product_purchase" <?php echo $type_filter == 'product_purchase' ? 'selected' : ''; ?>>Product Purchase</option>
                            </select>
                        </div>
                        
                        <!-- Start Date -->
                        <div>
                            <label class="block text-gray-700 text-sm font-bold mb-2">
                                From Date
                            </label>
                            <input type="date" name="start_date" value="<?php echo $start_date; ?>"
                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300">
                        </div>
                        
                        <!-- End Date -->
                        <div>
                            <label class="block text-gray-700 text-sm font-bold mb-2">
                                To Date
                            </label>
                            <input type="date" name="end_date" value="<?php echo $end_date; ?>"
                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300">
                        </div>
                    </div>
                    
                    <!-- Action Buttons -->
                    <div class="flex flex-wrap gap-3">
                        <button type="submit" 
                                class="bg-gradient-to-r from-blue-600 to-indigo-600 text-white font-bold py-3 px-6 rounded-lg hover:from-blue-700 hover:to-indigo-700 transition duration-300 transform hover:scale-105">
                            <i class="fas fa-filter mr-2"></i> Apply Filters
                        </button>
                        
                        <?php if($type_filter || $status_filter || $search || $start_date || $end_date): ?>
                            <a href="transactions.php" 
                               class="bg-gray-200 text-gray-700 font-bold py-3 px-6 rounded-lg hover:bg-gray-300 transition duration-300">
                                <i class="fas fa-times mr-2"></i> Clear Filters
                            </a>
                        <?php endif; ?>
                        
                        <!-- Export Button -->
                        <button type="button" onclick="exportTransactions()"
                                class="bg-gradient-to-r from-green-500 to-emerald-600 text-white font-bold py-3 px-6 rounded-lg hover:from-green-600 hover:to-emerald-700 transition duration-300">
                            <i class="fas fa-file-export mr-2"></i> Export CSV
                        </button>
                    </div>
                </form>
            </div>
            
            <!-- Transactions Table -->
            <div class="bg-white rounded-xl shadow-lg overflow-hidden">
                <div class="px-6 py-4 border-b bg-gradient-to-r from-blue-50 to-indigo-50">
                    <div class="flex items-center justify-between">
                        <h3 class="text-xl font-bold text-gray-800">
                            <i class="fas fa-list mr-2"></i> Transactions (<?php echo $total_transactions; ?>)
                        </h3>
                        <span class="text-gray-600 text-sm">
                            Page <?php echo $page; ?> of <?php echo max(1, $total_pages); ?>
                        </span>
                    </div>
                </div>
                
                <div class="overflow-x-auto">
                    <table class="w-full">
                        <thead>
                            <tr class="bg-gray-50">
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">ID</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Type</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Amount</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Description</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray-200">
                            <?php if($transactions_result->num_rows > 0): ?>
                                <?php while($transaction = $transactions_result->fetch_assoc()): 
                                    $type_color = match($transaction['type']) {
                                        'deposit' => 'green',
                                        'withdrawal' => 'red',
                                        'activation' => 'yellow',
                                        'referral_bonus' => 'blue',
                                        'product_purchase' => 'purple',
                                        default => 'gray'
                                    };
                                    
                                    $type_icon = match($transaction['type']) {
                                        'deposit' => 'fa-plus-circle',
                                        'withdrawal' => 'fa-minus-circle',
                                        'activation' => 'fa-bolt',
                                        'referral_bonus' => 'fa-users',
                                        'product_purchase' => 'fa-shopping-cart',
                                        default => 'fa-exchange-alt'
                                    };
                                    
                                    $status_color = match($transaction['status']) {
                                        'completed' => 'green',
                                        'pending' => 'yellow',
                                        'failed' => 'red',
                                        default => 'gray'
                                    };
                                    
                                    $is_positive = in_array($transaction['type'], ['deposit', 'referral_bonus']);
                                ?>
                                <tr class="transaction-row hover:bg-gray-50">
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="text-sm font-medium text-gray-900">
                                            #<?php echo str_pad($transaction['id'], 6, '0', STR_PAD_LEFT); ?>
                                        </div>
                                    </td>
                                    
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="flex items-center">
                                            <div class="w-8 h-8 rounded-full flex items-center justify-center mr-3 bg-<?php echo $type_color; ?>-100">
                                                <i class="fas <?php echo $type_icon; ?> text-<?php echo $type_color; ?>-600"></i>
                                            </div>
                                            <span class="text-sm font-medium text-gray-900">
                                                <?php echo ucfirst(str_replace('_', ' ', $transaction['type'])); ?>
                                            </span>
                                        </div>
                                    </td>
                                    
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <span class="text-lg font-bold <?php echo $is_positive ? 'text-green-600' : 'text-red-600'; ?>">
                                            <?php echo $is_positive ? '+' : '-'; ?>৳<?php echo number_format($transaction['amount'], 2); ?>
                                        </span>
                                    </td>
                                    
                                    <td class="px-6 py-4">
                                        <div class="text-sm text-gray-900 max-w-xs truncate">
                                            <?php echo htmlspecialchars($transaction['description']); ?>
                                        </div>
                                    </td>
                                    
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <span class="px-3 py-1 inline-flex text-xs leading-5 font-semibold rounded-full bg-<?php echo $status_color; ?>-100 text-<?php echo $status_color; ?>-800">
                                            <?php echo ucfirst($transaction['status']); ?>
                                        </span>
                                    </td>
                                    
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                        <?php echo date('M d, Y h:i A', strtotime($transaction['created_at'])); ?>
                                    </td>
                                    
                                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                        <button onclick="viewTransactionDetails(<?php echo $transaction['id']; ?>)" 
                                                class="text-blue-600 hover:text-blue-900 mr-3">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                        <?php if($transaction['status'] == 'pending' && $transaction['type'] == 'deposit'): ?>
                                            <button onclick="cancelTransaction(<?php echo $transaction['id']; ?>)" 
                                                    class="text-red-600 hover:text-red-900">
                                                <i class="fas fa-times"></i>
                                            </button>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endwhile; ?>
                            <?php else: ?>
                                <tr>
                                    <td colspan="7" class="px-6 py-12 text-center">
                                        <div class="w-24 h-24 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
                                            <i class="fas fa-exchange-alt text-gray-400 text-3xl"></i>
                                        </div>
                                        <h3 class="text-xl font-bold text-gray-700 mb-2">No Transactions Found</h3>
                                        <p class="text-gray-600 mb-6">
                                            <?php echo $search || $type_filter || $status_filter ? 'No transactions match your search criteria.' : 'You haven\'t made any transactions yet.'; ?>
                                        </p>
                                        <div class="flex flex-col sm:flex-row justify-center gap-4">
                                            <?php if($search || $type_filter || $status_filter): ?>
                                                <a href="transactions.php" 
                                                   class="bg-gray-200 text-gray-700 font-bold py-2 px-6 rounded-lg hover:bg-gray-300 transition duration-300">
                                                    <i class="fas fa-redo mr-2"></i> Reset Filters
                                                </a>
                                            <?php endif; ?>
                                            <a href="deposit.php" 
                                               class="bg-gradient-to-r from-blue-600 to-indigo-600 text-white font-bold py-2 px-6 rounded-lg hover:from-blue-700 hover:to-indigo-700 transition duration-300">
                                                <i class="fas fa-plus-circle mr-2"></i> Make First Deposit
                                            </a>
                                        </div>
                                    </td>
                                </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
                
                <!-- Pagination -->
                <?php if($total_pages > 1): ?>
                <div class="px-6 py-4 border-t">
                    <div class="flex items-center justify-between">
                        <div class="text-sm text-gray-700">
                            Showing <span class="font-medium"><?php echo $offset + 1; ?></span> to 
                            <span class="font-medium"><?php echo min($offset + $per_page, $total_transactions); ?></span> of 
                            <span class="font-medium"><?php echo $total_transactions; ?></span> transactions
                        </div>
                        
                        <nav class="flex items-center space-x-2">
                            <?php if($page > 1): ?>
                                <a href="transactions.php?page=<?php echo $page-1; ?><?php echo $type_filter ? '&type='.$type_filter : ''; ?><?php echo $status_filter ? '&status='.$status_filter : ''; ?><?php echo $search ? '&search='.urlencode($search) : ''; ?><?php echo $start_date ? '&start_date='.$start_date : ''; ?><?php echo $end_date ? '&end_date='.$end_date : ''; ?>"
                                   class="px-3 py-2 rounded-lg bg-white border border-gray-300 hover:bg-gray-50 transition duration-300">
                                    <i class="fas fa-chevron-left"></i> Previous
                                </a>
                            <?php endif; ?>
                            
                            <?php 
                            $start = max(1, $page - 2);
                            $end = min($total_pages, $page + 2);
                            
                            for($i = $start; $i <= $end; $i++): 
                            ?>
                                <a href="transactions.php?page=<?php echo $i; ?><?php echo $type_filter ? '&type='.$type_filter : ''; ?><?php echo $status_filter ? '&status='.$status_filter : ''; ?><?php echo $search ? '&search='.urlencode($search) : ''; ?><?php echo $start_date ? '&start_date='.$start_date : ''; ?><?php echo $end_date ? '&end_date='.$end_date : ''; ?>"
                                   class="px-3 py-2 rounded-lg font-medium transition duration-300
                                          <?php echo $i == $page ? 'bg-blue-600 text-white' : 'bg-white border border-gray-300 hover:bg-gray-50 text-gray-700'; ?>">
                                    <?php echo $i; ?>
                                </a>
                            <?php endfor; ?>
                            
                            <?php if($page < $total_pages): ?>
                                <a href="transactions.php?page=<?php echo $page+1; ?><?php echo $type_filter ? '&type='.$type_filter : ''; ?><?php echo $status_filter ? '&status='.$status_filter : ''; ?><?php echo $search ? '&search='.urlencode($search) : ''; ?><?php echo $start_date ? '&start_date='.$start_date : ''; ?><?php echo $end_date ? '&end_date='.$end_date : ''; ?>"
                                   class="px-3 py-2 rounded-lg bg-white border border-gray-300 hover:bg-gray-50 transition duration-300">
                                    Next <i class="fas fa-chevron-right"></i>
                                </a>
                            <?php endif; ?>
                        </nav>
                    </div>
                </div>
                <?php endif; ?>
            </div>
            
            <!-- Transaction Summary -->
            <div class="mt-8 grid grid-cols-1 md:grid-cols-3 gap-6">
                <div class="bg-white rounded-xl shadow-lg p-6">
                    <h3 class="text-lg font-bold text-gray-800 mb-4">
                        <i class="fas fa-chart-pie mr-2"></i> Transaction Summary
                    </h3>
                    <div class="space-y-4">
                        <?php if($type_filter || $status_filter || $search || $start_date || $end_date): ?>
                            <div class="p-3 bg-blue-50 rounded-lg">
                                <p class="text-blue-700 text-sm">
                                    <i class="fas fa-info-circle mr-2"></i>
                                    Showing filtered results. 
                                    <a href="transactions.php" class="underline">View all transactions</a>
                                </p>
                            </div>
                        <?php endif; ?>
                        
                        <div class="space-y-3">
                            <div class="flex justify-between">
                                <span class="text-gray-600">Total Transactions:</span>
                                <span class="font-bold"><?php echo $total_transactions; ?></span>
                            </div>
                            <div class="flex justify-between">
                                <span class="text-gray-600">Current Page:</span>
                                <span class="font-bold"><?php echo $page; ?> of <?php echo $total_pages; ?></span>
                            </div>
                            <div class="flex justify-between">
                                <span class="text-gray-600">Results per page:</span>
                                <span class="font-bold"><?php echo $per_page; ?></span>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-xl shadow-lg p-6">
                    <h3 class="text-lg font-bold text-gray-800 mb-4">
                        <i class="fas fa-download mr-2"></i> Export Options
                    </h3>
                    <div class="space-y-3">
                        <button onclick="exportTransactions()" 
                                class="w-full flex items-center justify-center bg-gradient-to-r from-green-500 to-emerald-600 text-white font-bold py-3 px-4 rounded-lg hover:from-green-600 hover:to-emerald-700 transition duration-300">
                            <i class="fas fa-file-csv mr-2"></i> Export as CSV
                        </button>
                        <button onclick="printTransactions()" 
                                class="w-full flex items-center justify-center bg-gradient-to-r from-blue-500 to-indigo-600 text-white font-bold py-3 px-4 rounded-lg hover:from-blue-600 hover:to-indigo-700 transition duration-300">
                            <i class="fas fa-print mr-2"></i> Print Report
                        </button>
                        <button onclick="shareReport()" 
                                class="w-full flex items-center justify-center bg-gradient-to-r from-purple-500 to-pink-600 text-white font-bold py-3 px-4 rounded-lg hover:from-purple-600 hover:to-pink-700 transition duration-300">
                            <i class="fas fa-share-alt mr-2"></i> Share Summary
                        </button>
                    </div>
                </div>
                
                <div class="bg-white rounded-xl shadow-lg p-6">
                    <h3 class="text-lg font-bold text-gray-800 mb-4">
                        <i class="fas fa-question-circle mr-2"></i> Need Help?
                    </h3>
                    <div class="space-y-3">
                        <a href="https://t.me/support" target="_blank" 
                           class="flex items-center justify-center bg-blue-100 text-blue-700 font-bold py-3 px-4 rounded-lg hover:bg-blue-200 transition duration-300">
                            <i class="fab fa-telegram mr-2"></i> Transaction Support
                        </a>
                        <a href="deposit.php" 
                           class="flex items-center justify-center bg-green-100 text-green-700 font-bold py-3 px-4 rounded-lg hover:bg-green-200 transition duration-300">
                            <i class="fas fa-plus-circle mr-2"></i> Deposit Issues
                        </a>
                        <a href="withdraw.php" 
                           class="flex items-center justify-center bg-red-100 text-red-700 font-bold py-3 px-4 rounded-lg hover:bg-red-200 transition duration-300">
                            <i class="fas fa-minus-circle mr-2"></i> Withdrawal Issues
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Transaction Details Modal -->
    <div id="transactionModal" class="fixed inset-0 bg-black bg-opacity-50 z-50 hidden items-center justify-center p-4">
        <div class="bg-white rounded-2xl max-w-2xl w-full max-h-[90vh] overflow-hidden">
            <div class="p-4 border-b flex justify-between items-center">
                <h3 class="text-xl font-bold text-gray-800">Transaction Details</h3>
                <button onclick="closeTransactionModal()" class="text-gray-500 hover:text-gray-700">
                    <i class="fas fa-times text-xl"></i>
                </button>
            </div>
            <div id="transactionContent" class="p-6 overflow-y-auto max-h-[calc(90vh-80px)]">
                <!-- Content loaded via AJAX -->
            </div>
        </div>
    </div>
    
    <script>
    // View transaction details
    function viewTransactionDetails(transactionId) {
        fetch('ajax/transaction-details.php?id=' + transactionId)
            .then(response => response.text())
            .then(html => {
                document.getElementById('transactionContent').innerHTML = html;
                document.getElementById('transactionModal').classList.remove('hidden');
                document.getElementById('transactionModal').classList.add('flex');
            })
            .catch(error => {
                console.error('Error:', error);
                showNotification('Failed to load transaction details', 'error');
            });
    }
    
    function closeTransactionModal() {
        document.getElementById('transactionModal').classList.add('hidden');
        document.getElementById('transactionModal').classList.remove('flex');
    }
    
    // Cancel transaction
    function cancelTransaction(transactionId) {
        if(!confirm('Are you sure you want to cancel this transaction?')) {
            return;
        }
        
        fetch('ajax/cancel-transaction.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: 'transaction_id=' + transactionId
        })
        .then(response => response.json())
        .then(data => {
            if(data.success) {
                showNotification('Transaction cancelled successfully!', 'success');
                setTimeout(() => {
                    location.reload();
                }, 1500);
            } else {
                showNotification(data.message || 'Error cancelling transaction', 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification('Network error', 'error');
        });
    }
    
    // Export transactions
    function exportTransactions() {
        const params = new URLSearchParams();
        
        <?php if($type_filter): ?>params.append('type', '<?php echo $type_filter; ?>');<?php endif; ?>
        <?php if($status_filter): ?>params.append('status', '<?php echo $status_filter; ?>');<?php endif; ?>
        <?php if($search): ?>params.append('search', '<?php echo $search; ?>');<?php endif; ?>
        <?php if($start_date): ?>params.append('start_date', '<?php echo $start_date; ?>');<?php endif; ?>
        <?php if($end_date): ?>params.append('end_date', '<?php echo $end_date; ?>');<?php endif; ?>
        
        window.open('ajax/export-transactions.php?' + params.toString(), '_blank');
    }
    
    // Print transactions
    function printTransactions() {
        const params = new URLSearchParams();
        
        <?php if($type_filter): ?>params.append('type', '<?php echo $type_filter; ?>');<?php endif; ?>
        <?php if($status_filter): ?>params.append('status', '<?php echo $status_filter; ?>');<?php endif; ?>
        <?php if($search): ?>params.append('search', '<?php echo $search; ?>');<?php endif; ?>
        <?php if($start_date): ?>params.append('start_date', '<?php echo $start_date; ?>');<?php endif; ?>
        <?php if($end_date): ?>params.append('end_date', '<?php echo $end_date; ?>');<?php endif; ?>
        
        const printWindow = window.open('ajax/print-transactions.php?' + params.toString(), '_blank');
        printWindow.onload = function() {
            printWindow.print();
        };
    }
    
    // Share report
    function shareReport() {
        const summary = `Transaction Summary:
Total Transactions: ${<?php echo $total_transactions; ?>}
Total Deposits: ৳${<?php echo $stats['total_deposits'] ?? 0; ?>}
Total Withdrawals: ৳${<?php echo $stats['total_withdrawals'] ?? 0; ?>}
Referral Earnings: ৳${<?php echo $stats['total_referral_bonus'] ?? 0; ?>}`;
        
        if(navigator.share) {
            navigator.share({
                title: 'My Transaction Report - Project Sale',
                text: summary,
                url: window.location.href
            });
        } else {
            // Fallback: Copy to clipboard
            navigator.clipboard.writeText(summary + '\n\nView full report: ' + window.location.href)
                .then(() => {
                    showNotification('Report copied to clipboard!', 'success');
                })
                .catch(() => {
                    showNotification('Failed to share report', 'error');
                });
        }
    }
    
    function showNotification(message, type) {
        const notification = document.createElement('div');
        notification.className = `fixed top-4 right-4 px-6 py-3 rounded-lg shadow-xl text-white font-bold z-50 animate__animated animate__fadeInRight ${type === 'success' ? 'bg-green-500' : 'bg-red-500'}`;
        notification.innerHTML = `
            <div class="flex items-center">
                <i class="fas ${type === 'success' ? 'fa-check-circle' : 'fa-exclamation-circle'} mr-3 text-xl"></i>
                <span>${message}</span>
            </div>
        `;
        document.body.appendChild(notification);
        
        setTimeout(() => {
            notification.classList.remove('animate__fadeInRight');
            notification.classList.add('animate__fadeOutRight');
            setTimeout(() => notification.remove(), 300);
        }, 3000);
    }
    
    // Add animations
    document.addEventListener('DOMContentLoaded', function() {
        const cards = document.querySelectorAll('.stat-card');
        cards.forEach((card, index) => {
            setTimeout(() => {
                card.classList.add('animate__animated', 'animate__fadeInUp');
            }, index * 100);
        });
        
        // Close modal when clicking outside
        document.addEventListener('click', function(event) {
            const modal = document.getElementById('transactionModal');
            const modalContent = modal.querySelector('.bg-white');
            
            if(modal.classList.contains('flex') && !modalContent.contains(event.target) && !event.target.closest('button[onclick*="viewTransactionDetails"]')) {
                closeTransactionModal();
            }
        });
    });
    </script>
</body>
</html>